(function () {
  function el(id) {
    return document.getElementById(id);
  }

  function showMessage(text, type) {
    var box = el('upword-connect-message');
    if (!box) return;
    box.style.display = 'block';
    box.className = 'upword-connect__message upword-connect__message--' + (type || 'info');
    box.textContent = text;
  }

  async function postJson(url, body) {
    var res = await fetch(url, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'X-WP-Nonce': (window.UpwordConnect && UpwordConnect.nonce) || ''
      },
      body: JSON.stringify(body || {})
    });
    var json = null;
    try { json = await res.json(); } catch (e) {}
    return { ok: res.ok, status: res.status, json: json };
  }

  async function startConnect() {
    var restBase = (window.UpwordConnect && UpwordConnect.restUrl) ? UpwordConnect.restUrl.replace(/\/$/, '') : '';
    if (!restBase) return;

    showMessage('Generating connect link…', 'info');
    var res = await postJson(restBase + '/connect/start', {});
    if (!res.ok || !res.json || !res.json.ok) {
      var msg = (res.json && res.json.error && res.json.error.message) ? res.json.error.message : 'Failed to start connect flow.';
      showMessage(msg, 'error');
      return;
    }

    var url = res.json.connect_url;
    if (!url) {
      showMessage('Connect URL was not returned.', 'error');
      return;
    }

    window.open(url, '_blank', 'noopener');
    showMessage('Upword opened in a new tab. Finish connecting there, then refresh this page.', 'success');
  }

  async function disconnect() {
    var restBase = (window.UpwordConnect && UpwordConnect.restUrl) ? UpwordConnect.restUrl.replace(/\/$/, '') : '';
    if (!restBase) {
      showMessage('REST API not available.', 'error');
      return;
    }
    if (!confirm('Disconnect Upword? This will stop automated publishing.')) return;

    showMessage('Disconnecting…', 'info');
    try {
      var res = await postJson(restBase + '/disconnect', {});
      if (!res.ok) {
        var errorMsg = (res.json && res.json.error && res.json.error.message) ? res.json.error.message : 'Failed to disconnect.';
        showMessage(errorMsg, 'error');
        console.error('Disconnect error:', res);
        return;
      }
      if (!res.json || !res.json.ok) {
        var errorMsg = (res.json && res.json.error && res.json.error.message) ? res.json.error.message : 'Disconnect failed.';
        showMessage(errorMsg, 'error');
        console.error('Disconnect failed:', res.json);
        return;
      }
      showMessage('Disconnected. Refreshing…', 'success');
      setTimeout(function() {
        window.location.reload();
      }, 500);
    } catch (err) {
      showMessage('Network error: ' + (err.message || 'Failed to disconnect.'), 'error');
      console.error('Disconnect exception:', err);
    }
  }

  async function saveSettings() {
    var restBase = (window.UpwordConnect && UpwordConnect.restUrl) ? UpwordConnect.restUrl.replace(/\/$/, '') : '';
    var sel = el('upword-seo-provider');
    var hosts = el('upword-allowed-image-hosts');
    var connectUrl = el('upword-connect-url');
    var maxMb = el('upword-max-image-mb');
    if (!restBase || !sel) return;

    showMessage('Saving settings…', 'info');
    var res = await postJson(restBase + '/settings', {
      seo_provider_preference: sel.value,
      allowed_image_hosts: hosts ? hosts.value : undefined,
      upword_connect_url: connectUrl ? connectUrl.value : undefined,
      max_image_mb: maxMb ? maxMb.value : undefined
    });

    if (!res.ok || !res.json || !res.json.ok) {
      showMessage('Failed to save settings.', 'error');
      return;
    }
    showMessage('Settings saved.', 'success');
  }

  document.addEventListener('DOMContentLoaded', function () {
    var btn = el('upword-connect-btn');
    if (btn) btn.addEventListener('click', startConnect);

    var dis = el('upword-disconnect-btn');
    if (dis) dis.addEventListener('click', disconnect);

    var save = el('upword-save-settings');
    if (save) save.addEventListener('click', saveSettings);
  });
})();

