<?php

if (!defined('ABSPATH')) {
  exit;
}

final class Upword_Connect_Admin {
  public static function init() {
    add_action('admin_menu', array(__CLASS__, 'register_menu'));
    add_action('admin_enqueue_scripts', array(__CLASS__, 'enqueue_assets'));
  }

  public static function register_menu() {
    add_options_page(
      __('Upword Connect', 'upword-connect'),
      __('Upword Connect', 'upword-connect'),
      'manage_options',
      'upword-connect',
      array(__CLASS__, 'render_page')
    );
  }

  public static function enqueue_assets($hook) {
    if ($hook !== 'settings_page_upword-connect') return;

    wp_enqueue_style(
      'upword-connect-admin',
      UPWORD_CONNECT_PLUGIN_URL . 'assets/css/admin.css',
      array(),
      UPWORD_CONNECT_VERSION
    );

    wp_enqueue_script(
      'upword-connect-admin',
      UPWORD_CONNECT_PLUGIN_URL . 'assets/js/admin.js',
      array('wp-api'),
      UPWORD_CONNECT_VERSION,
      true
    );

    wp_localize_script('upword-connect-admin', 'UpwordConnect', array(
      'restUrl' => esc_url_raw(rest_url('upword/v1')),
      'nonce' => wp_create_nonce('wp_rest'),
    ));
  }

  private static function esc($v) {
    return esc_html(is_scalar($v) ? (string) $v : '');
  }

  public static function render_page() {
    if (!current_user_can('manage_options')) return;
    $opts = Upword_Connect::get_options();
    $connected = !empty($opts['connected']);
    $site_url = isset($opts['site_url']) ? (string) $opts['site_url'] : home_url('/');
    $pref = isset($opts['seo_provider_preference']) ? (string) $opts['seo_provider_preference'] : 'auto';
    $last_error = isset($opts['last_error']) ? (string) $opts['last_error'] : '';
    $last_publish_at = isset($opts['last_publish_at']) ? (string) $opts['last_publish_at'] : '';
    $last_publish_status = isset($opts['last_publish_status']) ? (string) $opts['last_publish_status'] : '';
    $upword_connect_url = isset($opts['upword_connect_url']) ? (string) $opts['upword_connect_url'] : '';
    if (!$upword_connect_url) $upword_connect_url = 'https://www.goupword.com/connect/wordpress';
    $allowed_image_hosts = isset($opts['allowed_image_hosts']) ? (string) $opts['allowed_image_hosts'] : 'cdn.goupword.com';
    $max_image_mb = isset($opts['max_image_mb']) ? (int) $opts['max_image_mb'] : 10;

    $logo_url = UPWORD_CONNECT_PLUGIN_URL . 'assets/images/logo.svg';
    ?>
    <div class="wrap upword-connect">
      <div class="upword-connect__hero">
        <img class="upword-connect__logo" src="<?php echo esc_url($logo_url); ?>" alt="<?php echo esc_attr__('Upword', 'upword-connect'); ?>" />
        <div class="upword-connect__hero-text">
          <h1 class="upword-connect__title"><?php echo esc_html__('WordPress Publishing', 'upword-connect'); ?></h1>
          <p class="upword-connect__subtitle">
            <?php echo esc_html__('Connect your WordPress site to Upword to publish articles automatically.', 'upword-connect'); ?>
          </p>
        </div>
      </div>

      <div class="upword-connect__card">
        <div class="upword-connect__row">
          <div class="upword-connect__label"><?php echo esc_html__('Status', 'upword-connect'); ?></div>
          <div class="upword-connect__value">
            <?php if ($connected): ?>
              <span class="upword-connect__badge upword-connect__badge--ok"><?php echo esc_html__('Connected', 'upword-connect'); ?></span>
            <?php else: ?>
              <span class="upword-connect__badge upword-connect__badge--warn"><?php echo esc_html__('Not connected', 'upword-connect'); ?></span>
            <?php endif; ?>
          </div>
        </div>

        <div class="upword-connect__row">
          <div class="upword-connect__label"><?php echo esc_html__('Site URL', 'upword-connect'); ?></div>
          <div class="upword-connect__value"><?php echo esc_html($site_url); ?></div>
        </div>

        <div class="upword-connect__row">
          <div class="upword-connect__label"><?php echo esc_html__('SEO Provider', 'upword-connect'); ?></div>
          <div class="upword-connect__value">
            <select id="upword-seo-provider" class="upword-connect__select" data-current="<?php echo esc_attr($pref); ?>">
              <option value="auto" <?php selected($pref, 'auto'); ?>>Auto-detect</option>
              <option value="yoast" <?php selected($pref, 'yoast'); ?>>Yoast</option>
              <option value="rankmath" <?php selected($pref, 'rankmath'); ?>>Rank Math</option>
              <option value="custom" <?php selected($pref, 'custom'); ?>>Custom fields only</option>
            </select>
            <button type="button" class="button button-secondary" id="upword-save-settings">
              <?php echo esc_html__('Save', 'upword-connect'); ?>
            </button>
          </div>
        </div>

        <details class="upword-connect__advanced">
          <summary class="upword-connect__advanced-summary">
            <?php echo esc_html__('Advanced settings', 'upword-connect'); ?>
          </summary>
          <div class="upword-connect__advanced-body">
            <div class="upword-connect__row">
              <div class="upword-connect__label"><?php echo esc_html__('Allowed image hosts', 'upword-connect'); ?></div>
              <div class="upword-connect__value">
                <input
                  type="text"
                  id="upword-allowed-image-hosts"
                  class="regular-text"
                  value="<?php echo esc_attr($allowed_image_hosts); ?>"
                  placeholder="cdn.goupword.com, images.example.com"
                />
                <p class="description">
                  <?php echo esc_html__('Security: featured images will only be downloaded from these hostnames. Comma-separated.', 'upword-connect'); ?>
                </p>
              </div>
            </div>

            <div class="upword-connect__row">
              <div class="upword-connect__label"><?php echo esc_html__('Max image size (MB)', 'upword-connect'); ?></div>
              <div class="upword-connect__value">
                <input
                  type="number"
                  id="upword-max-image-mb"
                  min="1"
                  max="50"
                  value="<?php echo esc_attr((string) $max_image_mb); ?>"
                  style="width:120px;"
                />
              </div>
            </div>

            <div class="upword-connect__row">
              <div class="upword-connect__label"><?php echo esc_html__('Upword connect URL', 'upword-connect'); ?></div>
              <div class="upword-connect__value">
                <input
                  type="url"
                  id="upword-connect-url"
                  class="regular-text"
                  value="<?php echo esc_attr($upword_connect_url); ?>"
                />
                <p class="description">
                  <?php echo esc_html__('Advanced: only change this if Upword instructs you to (e.g., staging).', 'upword-connect'); ?>
                </p>
              </div>
            </div>
          </div>
        </details>

        <?php if ($last_error): ?>
          <div class="upword-connect__row">
            <div class="upword-connect__label"><?php echo esc_html__('Last error', 'upword-connect'); ?></div>
            <div class="upword-connect__value upword-connect__value--error"><?php echo esc_html($last_error); ?></div>
          </div>
        <?php endif; ?>

        <?php if ($last_publish_at): ?>
          <div class="upword-connect__row">
            <div class="upword-connect__label"><?php echo esc_html__('Last publish', 'upword-connect'); ?></div>
            <div class="upword-connect__value">
              <?php echo esc_html($last_publish_at); ?>
              <?php if ($last_publish_status): ?>
                <span class="upword-connect__muted">(<?php echo esc_html($last_publish_status); ?>)</span>
              <?php endif; ?>
            </div>
          </div>
        <?php endif; ?>

        <div class="upword-connect__actions">
          <button type="button" class="button button-primary" id="upword-connect-btn" data-upword-connect-url="<?php echo esc_attr($upword_connect_url); ?>">
            <?php echo esc_html__('Connect to Upword', 'upword-connect'); ?>
          </button>

          <?php if ($connected): ?>
            <button type="button" class="button button-secondary" id="upword-disconnect-btn">
              <?php echo esc_html__('Disconnect', 'upword-connect'); ?>
            </button>
          <?php endif; ?>

          <span class="upword-connect__hint">
            <?php echo esc_html__('This opens Upword in a new tab. After you log in and approve, come back here to see “Connected”.', 'upword-connect'); ?>
          </span>
        </div>

        <div id="upword-connect-message" class="upword-connect__message" style="display:none;"></div>
      </div>
    </div>
    <?php
  }
}

