<?php

if (!defined('ABSPATH')) {
  exit;
}

final class Upword_Connect_Rest {
  const NS = 'upword/v1';
  const TRANSIENT_PREFIX = 'upword_connect_';
  const CONNECT_TTL_SECONDS = 600; // 10 minutes
  const HMAC_DRIFT_SECONDS = 300; // 5 minutes
  const MAX_HTML_CHARS = 2000000; // 2MB-ish
  const MAX_EXCERPT_CHARS = 10000;

  public static function init() {
    add_action('rest_api_init', array(__CLASS__, 'register_routes'));
  }

  public static function register_routes() {
    register_rest_route(self::NS, '/connect/start', array(
      'methods' => 'POST',
      'callback' => array(__CLASS__, 'connect_start'),
      'permission_callback' => array(__CLASS__, 'require_admin'),
    ));

    register_rest_route(self::NS, '/settings', array(
      'methods' => 'POST',
      'callback' => array(__CLASS__, 'update_settings'),
      'permission_callback' => array(__CLASS__, 'require_admin'),
    ));

    register_rest_route(self::NS, '/connect/confirm', array(
      'methods' => 'POST',
      'callback' => array(__CLASS__, 'connect_confirm'),
      'permission_callback' => '__return_true', // secured by connect_code + state
    ));

    register_rest_route(self::NS, '/publish', array(
      'methods' => 'POST',
      'callback' => array(__CLASS__, 'publish'),
      'permission_callback' => '__return_true', // secured by HMAC
    ));

    register_rest_route(self::NS, '/disconnect', array(
      'methods' => 'POST',
      'callback' => array(__CLASS__, 'disconnect'),
      'permission_callback' => array(__CLASS__, 'disconnect_permission'),
    ));
  }

  public static function require_admin() {
    return current_user_can('manage_options');
  }

  public static function disconnect_permission(WP_REST_Request $request) {
    // Allow either admin user OR HMAC-authenticated request from Upword
    if (current_user_can('manage_options')) {
      return true;
    }
    
    // Check for HMAC authentication (from Upword server)
    $raw_body = $request->get_body();
    $ts = (string) $request->get_header('x-upword-timestamp');
    $sig = (string) $request->get_header('x-upword-signature');
    
    if ($ts && $sig) {
      $opts = Upword_Connect::get_options();
      $secret = isset($opts['shared_secret']) ? (string) $opts['shared_secret'] : '';
      if ($secret) {
        if (!ctype_digit($ts)) {
          return false;
        }
        $now = time();
        $t = (int) $ts;
        if (abs($now - $t) > self::HMAC_DRIFT_SECONDS) {
          return false;
        }
        $computed = hash_hmac('sha256', $ts . '.' . $raw_body, $secret);
        if (hash_equals($computed, $sig)) {
          return true; // HMAC-authenticated request from Upword
        }
      }
    }
    
    return false;
  }

  private static function json_error($code, $message, $status = 400) {
    // Persist last error for admin visibility (best effort).
    if (in_array((int) $status, array(400, 401, 403, 429, 500, 502), true)) {
      Upword_Connect::update_options(array(
        'last_error' => (string) $code . ': ' . (string) $message,
        'last_publish_status' => in_array((string) $code, array('publish_failed', 'invalid_signature', 'invalid_payload'), true) ? 'failed' : null,
      ));
    }
    return new WP_REST_Response(array(
      'ok' => false,
      'error' => array(
        'code' => (string) $code,
        'message' => (string) $message,
      ),
    ), (int) $status);
  }

  private static function get_client_ip() {
    $ip = '';
    if (!empty($_SERVER['REMOTE_ADDR'])) {
      $ip = (string) $_SERVER['REMOTE_ADDR'];
    }
    $ip = trim($ip);
    return $ip ? $ip : 'unknown';
  }

  private static function rate_limit($bucket, $limit, $window_seconds) {
    $ip = self::get_client_ip();
    $key = 'upword_rl_' . md5($bucket . '|' . $ip);
    $data = get_transient($key);
    if (!is_array($data)) {
      $data = array('count' => 0, 'reset' => time() + (int) $window_seconds);
    }

    $now = time();
    if (!empty($data['reset']) && (int) $data['reset'] < $now) {
      $data = array('count' => 0, 'reset' => $now + (int) $window_seconds);
    }

    $data['count'] = (int) $data['count'] + 1;
    set_transient($key, $data, (int) $window_seconds);

    if ((int) $data['count'] > (int) $limit) {
      return false;
    }
    return true;
  }

  private static function random_token($prefix, $bytes = 18) {
    try {
      $raw = random_bytes((int) $bytes);
    } catch (Exception $e) {
      $raw = wp_generate_password((int) $bytes * 2, false, false);
      return $prefix . $raw;
    }
    return $prefix . bin2hex($raw);
  }

  private static function detect_seo_provider() {
    if (defined('WPSEO_VERSION') || class_exists('WPSEO_Options')) {
      return 'yoast';
    }
    if (defined('RANK_MATH_VERSION') || class_exists('RankMath\\Helper')) {
      return 'rankmath';
    }
    return 'custom';
  }

  public static function connect_start(WP_REST_Request $request) {
    $opts = Upword_Connect::get_options();
    $base = isset($opts['upword_connect_url']) ? trim((string) $opts['upword_connect_url']) : '';
    if (!$base) {
      $base = 'https://www.goupword.com/connect/wordpress';
    }

    $connect_code = self::random_token('uwc_', 12);
    $state = self::random_token('st_', 12);

    $payload = array(
      'state' => $state,
      'created_at' => time(),
      'used' => false,
    );
    set_transient(self::TRANSIENT_PREFIX . $connect_code, $payload, self::CONNECT_TTL_SECONDS);

    $site_url = home_url('/');
    $expires_at = gmdate('c', time() + self::CONNECT_TTL_SECONDS);

    $connect_url = add_query_arg(array(
      'site_url' => $site_url,
      'connect_code' => $connect_code,
      'state' => $state,
    ), $base);

    return new WP_REST_Response(array(
      'ok' => true,
      'connect_code' => $connect_code,
      'state' => $state,
      'expires_at' => $expires_at,
      'site_url' => $site_url,
      'connect_url' => $connect_url,
    ), 200);
  }

  public static function connect_confirm(WP_REST_Request $request) {
    // Throttle brute-force attempts.
    if (!self::rate_limit('connect_confirm', 30, 60)) {
      return self::json_error('rate_limited', 'Too many requests.', 429);
    }

    $body = $request->get_json_params();
    $connect_code = isset($body['connect_code']) ? trim((string) $body['connect_code']) : '';
    $state = isset($body['state']) ? trim((string) $body['state']) : '';
    $upword_client_id = isset($body['upword_client_id']) ? trim((string) $body['upword_client_id']) : '';
    $upword_api_base_url = isset($body['upword_api_base_url']) ? trim((string) $body['upword_api_base_url']) : '';

    if (!$connect_code || !$state) {
      return self::json_error('invalid_payload', 'connect_code and state are required', 400);
    }

    $key = self::TRANSIENT_PREFIX . $connect_code;
    $data = get_transient($key);
    if (!is_array($data)) {
      return self::json_error('invalid_connect_code', 'Connect code is invalid or expired.', 400);
    }

    if (!empty($data['used'])) {
      return self::json_error('invalid_connect_code', 'Connect code has already been used.', 400);
    }

    if (!isset($data['state']) || !hash_equals((string) $data['state'], (string) $state)) {
      return self::json_error('state_mismatch', 'State does not match.', 400);
    }

    // Single-use: delete immediately to prevent replay.
    delete_transient($key);

    $shared_secret = self::random_token('sec_', 32);
    $site_url = home_url('/');
    $seo = self::detect_seo_provider();

    // Store Upword client_id and API base URL for bidirectional sync
    $connection_data = array(
      'shared_secret' => $shared_secret,
      'site_url' => $site_url,
      'seo_provider_detected' => $seo
    );
    
    if ($upword_client_id) {
      $connection_data['upword_client_id'] = $upword_client_id;
    }
    
    if ($upword_api_base_url) {
      $connection_data['upword_api_base_url'] = $upword_api_base_url;
    }

    Upword_Connect::set_connected($shared_secret, $site_url, $seo, $connection_data);

    return new WP_REST_Response(array(
      'ok' => true,
      'api_version' => '2026-01-01',
      'site' => array(
        'url' => $site_url,
        'name' => get_bloginfo('name'),
        'wp_version' => get_bloginfo('version'),
        'plugin_version' => defined('UPWORD_CONNECT_VERSION') ? UPWORD_CONNECT_VERSION : null,
      ),
      'seo_provider_detected' => $seo,
      'shared_secret' => $shared_secret,
    ), 200);
  }

  public static function update_settings(WP_REST_Request $request) {
    $body = $request->get_json_params();
    if (!is_array($body)) {
      return self::json_error('invalid_payload', 'Invalid JSON.', 400);
    }

    $patch = array();

    if (isset($body['seo_provider_preference'])) {
      $pref = strtolower(trim((string) $body['seo_provider_preference']));
      if (!in_array($pref, array('auto', 'yoast', 'rankmath', 'custom'), true)) {
        return self::json_error('invalid_payload', 'Invalid seo_provider_preference.', 400);
      }
      $patch['seo_provider_preference'] = $pref;
    }

    if (isset($body['target_post_type'])) {
      $pt = sanitize_key((string) $body['target_post_type']);
      if ($pt) $patch['target_post_type'] = $pt;
    }

    if (isset($body['upword_connect_url'])) {
      $u = esc_url_raw((string) $body['upword_connect_url']);
      // Allow both http:// and https:// (http:// for local development)
      if ($u && (stripos($u, 'https://') === 0 || stripos($u, 'http://') === 0)) {
        $patch['upword_connect_url'] = $u;
      }
    }

    if (isset($body['allowed_image_hosts'])) {
      $raw = (string) $body['allowed_image_hosts'];
      $parts = preg_split('/[,\s]+/', $raw);
      $clean = array();
      if (is_array($parts)) {
        foreach ($parts as $p) {
          $p = strtolower(trim((string) $p));
          if (!$p) continue;
          // Allow wildcards like *.example.com
          $p = preg_replace('/[^a-z0-9\.\-\*]/', '', $p);
          if (!$p) continue;
          $clean[] = $p;
        }
      }
      $clean = array_values(array_unique($clean));
      if (!empty($clean)) {
        $patch['allowed_image_hosts'] = implode(', ', $clean);
      }
    }

    if (isset($body['max_image_mb'])) {
      $mb = (int) $body['max_image_mb'];
      if ($mb >= 1 && $mb <= 50) $patch['max_image_mb'] = $mb;
    }

    if (isset($body['upword_api_base_url'])) {
      $u = esc_url_raw((string) $body['upword_api_base_url']);
      // Allow both http:// and https:// (http:// for local development)
      if ($u && (stripos($u, 'https://') === 0 || stripos($u, 'http://') === 0)) {
        $patch['upword_api_base_url'] = $u;
      }
    }

    if (empty($patch)) {
      return new WP_REST_Response(array('ok' => true), 200);
    }

    Upword_Connect::update_options($patch);
    return new WP_REST_Response(array('ok' => true), 200);
  }

  private static function verify_hmac(WP_REST_Request $request, $raw_body) {
    $opts = Upword_Connect::get_options();
    $secret = isset($opts['shared_secret']) ? (string) $opts['shared_secret'] : '';
    if (!$secret) {
      return self::json_error('not_connected', 'Site is not connected to Upword.', 401);
    }

    $ts = (string) $request->get_header('x-upword-timestamp');
    $sig = (string) $request->get_header('x-upword-signature');
    if (!$ts || !$sig) {
      return self::json_error('invalid_signature', 'Missing signature headers.', 401);
    }

    if (!ctype_digit($ts)) {
      return self::json_error('invalid_signature', 'Invalid timestamp.', 401);
    }

    $now = time();
    $t = (int) $ts;
    if (abs($now - $t) > self::HMAC_DRIFT_SECONDS) {
      return self::json_error('invalid_signature', 'Timestamp drift too large.', 401);
    }

    $computed = hash_hmac('sha256', $ts . '.' . $raw_body, $secret);
    if (!hash_equals($computed, $sig)) {
      return self::json_error('invalid_signature', 'Signature verification failed.', 401);
    }

    return true;
  }

  private static function normalize_status($status) {
    $s = strtolower(trim((string) $status));
    if ($s === 'publish' || $s === 'published') return 'publish';
    if ($s === 'draft') return 'draft';
    return 'draft';
  }

  private static function get_target_post_type() {
    $opts = Upword_Connect::get_options();
    $pt = isset($opts['target_post_type']) ? trim((string) $opts['target_post_type']) : 'post';
    return $pt ? $pt : 'post';
  }

  private static function find_post_by_idempotency($idempotency_key, $post_type) {
    $q = new WP_Query(array(
      'post_type' => $post_type,
      'post_status' => 'any',
      'posts_per_page' => 1,
      'fields' => 'ids',
      'no_found_rows' => true,
      'meta_query' => array(
        array(
          'key' => 'upword_idempotency_key',
          'value' => (string) $idempotency_key,
          'compare' => '=',
        ),
      ),
    ));
    return !empty($q->posts) ? (int) $q->posts[0] : 0;
  }

  private static function is_host_allowed($host) {
    $host = strtolower(trim((string) $host));
    if (!$host) return false;

    $opts = Upword_Connect::get_options();
    $allowed = isset($opts['allowed_image_hosts']) ? (string) $opts['allowed_image_hosts'] : 'cdn.goupword.com';
    $parts = preg_split('/[,\s]+/', $allowed);
    if (!is_array($parts) || empty($parts)) return false;

    foreach ($parts as $p) {
      $p = strtolower(trim((string) $p));
      if (!$p) continue;
      if ($p === $host) return true;
      if (strpos($p, '*.') === 0) {
        $suffix = substr($p, 1); // ".example.com"
        if ($suffix && substr($host, -strlen($suffix)) === $suffix) return true;
      }
    }
    return false;
  }

  private static function is_ip_public($ip) {
    return (bool) filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE);
  }

  private static function validate_image_url($url) {
    $url = trim((string) $url);
    if (!$url) return false;
    if (stripos($url, 'https://') !== 0) return false;
    if (!wp_http_validate_url($url)) return false;

    $parts = wp_parse_url($url);
    if (!is_array($parts) || empty($parts['host'])) return false;
    $host = strtolower((string) $parts['host']);
    if (!self::is_host_allowed($host)) return false;

    $ips = @gethostbynamel($host);
    if (is_array($ips) && !empty($ips)) {
      foreach ($ips as $ip) {
        if (!self::is_ip_public($ip)) return false;
      }
    }
    return true;
  }

  private static function download_image_to_tempfile($image_url, &$filename_out, &$error_out) {
    $error_out = '';
    $filename_out = '';

    if (!self::validate_image_url($image_url)) {
      $error_out = 'Image URL is not allowed.';
      return false;
    }

    $name = wp_basename(parse_url($image_url, PHP_URL_PATH));
    if (!$name) $name = 'upword-image.jpg';
    $filename_out = $name;

    $tmp = wp_tempnam($name);
    if (!$tmp) {
      $error_out = 'Failed to create temp file.';
      return false;
    }

    $opts = Upword_Connect::get_options();
    $max_mb = isset($opts['max_image_mb']) ? (int) $opts['max_image_mb'] : 10;
    $max_bytes = max(1, $max_mb) * 1024 * 1024;

    $res = wp_safe_remote_get($image_url, array(
      'timeout' => 15,
      'redirection' => 3,
      'stream' => true,
      'filename' => $tmp,
      'limit_response_size' => $max_bytes,
      'reject_unsafe_urls' => true,
    ));

    if (is_wp_error($res)) {
      $error_out = $res->get_error_message();
      unlink($tmp);
      return false;
    }

    $code = (int) wp_remote_retrieve_response_code($res);
    if ($code < 200 || $code >= 300) {
      $error_out = 'Image download failed with HTTP ' . $code;
      unlink($tmp);
      return false;
    }

    $ctype = (string) wp_remote_retrieve_header($res, 'content-type');
    if ($ctype && stripos($ctype, 'image/') !== 0) {
      $error_out = 'Downloaded file is not an image.';
      unlink($tmp);
      return false;
    }

    return $tmp;
  }

  private static function maybe_import_featured_image($post_id, $image_url) {
    $image_url = trim((string) $image_url);
    if (!$image_url) return 0;

    $existing = (string) get_post_meta($post_id, 'upword_featured_image_source', true);
    if ($existing && hash_equals($existing, $image_url)) {
      $thumb_id = get_post_thumbnail_id($post_id);
      return $thumb_id ? (int) $thumb_id : 0;
    }

    require_once ABSPATH . 'wp-admin/includes/file.php';
    require_once ABSPATH . 'wp-admin/includes/media.php';
    require_once ABSPATH . 'wp-admin/includes/image.php';

    $name = '';
    $err = '';
    $tmp = self::download_image_to_tempfile($image_url, $name, $err);
    if (!$tmp) {
      Upword_Connect::update_options(array('last_error' => 'image_download_failed: ' . $err));
      return 0;
    }

    $file_array = array(
      'name' => $name,
      'tmp_name' => $tmp,
    );

    $attachment_id = media_handle_sideload($file_array, $post_id);
    unlink($tmp);

    if (is_wp_error($attachment_id)) {
      Upword_Connect::update_options(array('last_error' => 'image_import_failed: ' . $attachment_id->get_error_message()));
      return 0;
    }

    set_post_thumbnail($post_id, $attachment_id);
    update_post_meta($post_id, 'upword_featured_image_source', $image_url);

    return (int) $attachment_id;
  }

  private static function apply_seo_meta($post_id, $seo, $article_id, $idempotency_key) {
    $title = isset($seo['title']) ? (string) $seo['title'] : '';
    $desc = isset($seo['description']) ? (string) $seo['description'] : '';

    // Always write Upword custom fields.
    update_post_meta($post_id, 'upword_seo_title', $title);
    update_post_meta($post_id, 'upword_seo_description', $desc);
    update_post_meta($post_id, 'upword_article_id', (string) $article_id);
    update_post_meta($post_id, 'upword_idempotency_key', (string) $idempotency_key);

    $opts = Upword_Connect::get_options();
    $pref = isset($opts['seo_provider_preference']) ? strtolower((string) $opts['seo_provider_preference']) : 'auto';
    $detected = self::detect_seo_provider();

    $provider = $pref === 'auto' ? $detected : $pref;
    if (!in_array($provider, array('yoast', 'rankmath', 'custom'), true)) {
      $provider = 'custom';
    }

    if ($provider === 'yoast' || ($pref === 'auto' && $detected === 'yoast')) {
      if ($title) update_post_meta($post_id, '_yoast_wpseo_title', $title);
      if ($desc) update_post_meta($post_id, '_yoast_wpseo_metadesc', $desc);
      return 'yoast';
    }

    if ($provider === 'rankmath' || ($pref === 'auto' && $detected === 'rankmath')) {
      if ($title) {
        update_post_meta($post_id, 'rank_math_title', $title);
        update_post_meta($post_id, '_rank_math_title', $title);
      }
      if ($desc) {
        update_post_meta($post_id, 'rank_math_description', $desc);
        update_post_meta($post_id, '_rank_math_description', $desc);
      }
      return 'rankmath';
    }

    return 'custom';
  }

  public static function publish(WP_REST_Request $request) {
    // Throttle requests (defense-in-depth).
    if (!self::rate_limit('publish', 120, 60)) {
      return self::json_error('rate_limited', 'Too many requests.', 429);
    }

    $raw_body = file_get_contents('php://input');
    if ($raw_body === false) $raw_body = '';

    $auth = self::verify_hmac($request, $raw_body);
    if ($auth !== true) return $auth;

    $body = $request->get_json_params();
    if (!is_array($body)) {
      return self::json_error('invalid_payload', 'Invalid JSON.', 400);
    }

    $required = array('idempotency_key', 'article_id', 'title', 'slug', 'html', 'status');
    foreach ($required as $k) {
      if (!isset($body[$k]) || trim((string) $body[$k]) === '') {
        return self::json_error('invalid_payload', 'Missing required field: ' . $k, 400);
      }
    }

    $idempotency_key = (string) $body['idempotency_key'];
    $article_id = (string) $body['article_id'];
    $title = (string) $body['title'];
    $slug = (string) $body['slug'];
    $html = (string) $body['html'];
    $excerpt = isset($body['excerpt']) ? (string) $body['excerpt'] : '';
    $status = self::normalize_status($body['status']);
    $image_url = isset($body['image_url']) ? (string) $body['image_url'] : '';
    $seo = isset($body['seo']) && is_array($body['seo']) ? $body['seo'] : array();

    if (strlen($html) > self::MAX_HTML_CHARS) {
      return self::json_error('invalid_payload', 'html is too large.', 400);
    }
    if ($excerpt && strlen($excerpt) > self::MAX_EXCERPT_CHARS) {
      $excerpt = substr($excerpt, 0, self::MAX_EXCERPT_CHARS);
    }

    $json_ld = '';
    if (preg_match('/^(.*)<script[^>]*type=["\']application\/ld\+json["\'][^>]*>([\s\S]*?)<\/script>\s*$/s', $html, $m)) {
      $json_ld = trim($m[2]);
      $html = trim($m[1]);
    }
    if ($json_ld !== '') {
      $html .= "\n[upword_json_ld]";
    }

    $post_type = self::get_target_post_type();

    $post_id = self::find_post_by_idempotency($idempotency_key, $post_type);
    $action = $post_id ? 'updated' : 'created';

    $postarr = array(
      'post_type' => $post_type,
      'post_title' => wp_strip_all_tags($title),
      'post_content' => $html,
      'post_excerpt' => $excerpt,
      'post_status' => $status,
      'post_name' => sanitize_title($slug),
    );

    if ($post_id) {
      $postarr['ID'] = $post_id;
      $result = wp_update_post($postarr, true);
    } else {
      $result = wp_insert_post($postarr, true);
      if (!is_wp_error($result)) {
        $post_id = (int) $result;
      }
    }

    if (is_wp_error($result) || !$post_id) {
      return self::json_error('publish_failed', 'Failed to create/update post.', 500);
    }

    // Persist linkage and idempotency on the post.
    update_post_meta($post_id, 'upword_article_id', $article_id);
    update_post_meta($post_id, 'upword_idempotency_key', $idempotency_key);

    if ($json_ld !== '') {
      update_post_meta($post_id, 'upword_json_ld', $json_ld);
    } else {
      delete_post_meta($post_id, 'upword_json_ld');
    }

    $featured_image_id = self::maybe_import_featured_image($post_id, $image_url);
    $seo_provider_applied = self::apply_seo_meta($post_id, $seo, $article_id, $idempotency_key);

    Upword_Connect::update_options(array(
      'last_publish_at' => gmdate('c'),
      'last_publish_status' => 'ok',
      'last_error' => '',
    ));

    $url = get_permalink($post_id);

    return new WP_REST_Response(array(
      'ok' => true,
      'action' => $action,
      'wp_post_id' => $post_id,
      'wp_post_url' => $url ? (string) $url : null,
      'featured_image_id' => $featured_image_id ?: null,
      'seo_provider_applied' => $seo_provider_applied,
    ), 200);
  }

  public static function disconnect(WP_REST_Request $request) {
    // Permission check is handled by disconnect_permission callback
    // which allows both admin users and HMAC-authenticated requests from Upword
    
    // Determine disconnect source
    $is_from_upword = !empty($request->get_header('x-upword-signature'));
    $error_message = $is_from_upword ? 'Disconnected by Upword.' : 'Disconnected by admin.';
    
    // Get options BEFORE clearing connection (so we have API URL and client_id)
    $opts = Upword_Connect::get_options();
    $api_base = isset($opts['upword_api_base_url']) ? trim((string) $opts['upword_api_base_url']) : '';
    $client_id = isset($opts['upword_client_id']) ? trim((string) $opts['upword_client_id']) : '';
    
    // If disconnecting from WordPress admin (not from Upword), try to notify Upword
    if (!$is_from_upword && $api_base && $client_id) {
      // Try to notify Upword (non-fatal if it fails)
      $upword_disconnect_url = rtrim($api_base, '/') . '/api/integrations/wordpress/disconnect';
      $response = wp_remote_post($upword_disconnect_url, array(
        'timeout' => 10,
        'body' => json_encode(array('clientId' => $client_id)),
        'headers' => array('Content-Type' => 'application/json'),
      ));
      
      // Log errors but don't fail the disconnect
      if (is_wp_error($response)) {
        error_log('Upword disconnect notification failed: ' . $response->get_error_message());
      }
    }
    
    // Clear the connection
    $result = Upword_Connect::clear_connection($error_message);
    
    // Verify the connection was cleared
    $opts_after = Upword_Connect::get_options();
    if (!empty($opts_after['connected'])) {
      return self::json_error('disconnect_failed', 'Failed to clear connection.', 500);
    }
    
    return new WP_REST_Response(array('ok' => true), 200);
  }
}

